#include <WiFi.h>
#include <HTTPClient.h>
#include <Wire.h>
#include <Adafruit_GFX.h>
#include <Adafruit_SSD1306.h>
#include "SparkFun_CAP1203.h"

// ---------------- OLED ----------------
#define SCREEN_WIDTH 128
#define SCREEN_HEIGHT 64
#define OLED_RESET -1
#define OLED_ADDR 0x3C
#define DEBUG_API 1

Adafruit_SSD1306 display(SCREEN_WIDTH, SCREEN_HEIGHT, &Wire, OLED_RESET);

// ---------------- TOUCH ----------------
CAP1203 touch;

// ---------------- MARS COMPANION ----------------
const char* COMPANION_API = "http://192.168.0.118:5001/interact";

bool promptActive = false;
String activePrompt = "";
unsigned long lastCompanionPoll = 0;
unsigned long lastScreenChange = 0;
const unsigned long COMPANION_INTERVAL = 10000;
const unsigned long SCREEN_CHANGE_INTERVAL = 4000;


// ---------------- WIFI ----------------
const char* ssid = "Parsec-Guest";
const char* password = "Parsec@Guest";

// ---------------- API ----------------
const char* TIME_API = "http://192.168.0.118:5000/time";
const char* DHT_API  = "http://192.168.0.118:5000/dht22";

// ---------------- DATA ----------------
String earthTime = "--";
String marsTime  = "--";
float temperature = 0.0;
float humidity    = 0.0;

String userInput = "none";
String replyReceived = "none";

uint8_t currentScreen = 0;
unsigned long lastFetch = 0;
const unsigned long FETCH_INTERVAL = 5000;

// ---------------- HELPERS ----------------

void dbg(const String &label, const String &value) {
#if DEBUG_API
  Serial.print("[DBG] ");
  Serial.print(label);
  Serial.print(": ");
  Serial.println(value);
#endif
}

void dbgFloat(const String &label, float value) {
#if DEBUG_API
  Serial.print("[DBG] ");
  Serial.print(label);
  Serial.print(": ");
  Serial.println(value, 3);
#endif
}

String extractValue(String json, String key) {
  int i = json.indexOf(key);
  if (i == -1) return "";
  int c = json.indexOf(":", i);
  int s = json.indexOf("\"", c) + 1;
  int e = json.indexOf("\"", s);
  return json.substring(s, e);
}

float extractFloat(String json, String key) {
  int i = json.indexOf(key);
  if (i == -1) return 0;
  int c = json.indexOf(":", i);
  int e = json.indexOf(",", c);
  if (e == -1) e = json.indexOf("}", c);
  return json.substring(c + 1, e).toFloat();
}

String formatEarthTime(String iso) {
  if (iso.length() < 19) return iso;
  return iso.substring(0, 10) + " " + iso.substring(11, 19);
}

String extractRaw(String json, String key) {
  int i = json.indexOf(key);
  if (i == -1) return "";
  int c = json.indexOf(":", i);
  int s = json.indexOf("\"", c) + 1;
  int e = json.indexOf("\"", s);
  return json.substring(s, e);
}

String extractObject(String json, String key) {
  int i = json.indexOf(key);
  if (i == -1) return "";
  int c = json.indexOf(":", i);
  int s = json.indexOf("\"", c) + 1;
  int e = json.lastIndexOf("\"");
  return json.substring(s, e);
}



// ---------------- OLED ----------------
void drawOLED() {
  display.clearDisplay();
  display.setTextColor(SSD1306_WHITE);
  display.setTextSize(1);
  display.setCursor(0, 8);

  switch (currentScreen) {

    case 0:
      display.println("EARTH TIME");
      display.setTextSize(2);
      display.setCursor(0, 30);
      display.println(formatEarthTime(earthTime));
      break;

    case 1:
      display.println("MARS TIME");
      display.setTextSize(2);
      display.setCursor(0, 30);
      display.println(marsTime);
      break;

    case 2:
      display.println("TEMPERATURE");
      display.setTextSize(2);
      display.setCursor(0, 34);
      display.print(temperature, 1);
      display.print(" C");
      break;

    case 3:
      display.println("HUMIDITY");
      display.setTextSize(2);
      display.setCursor(0, 34);
      display.print(humidity, 1);
      display.print(" %");
      break;
    
    case 4:
      display.setTextSize(2);
      display.println("YES");
      break;

    case 5:
      display.setTextSize(2);
      display.println("NO");
      break;
    
    case 6:
      display.setTextSize(2);
      display.println("SOS");
      break;

  }


  display.display();
}

void drawPrompt(String text) {
  display.clearDisplay();
  display.setTextColor(SSD1306_WHITE);
  display.setTextSize(2);
  display.setCursor(0, 0);
  display.println(text);
  display.display();
}

// ---------------- API ----------------
void fetchTime() {
  HTTPClient http;
  http.begin(TIME_API);
  int code = http.GET();

#if DEBUG_API
  Serial.println("\n=== TIME API ===");
  Serial.print("HTTP Code: ");
  Serial.println(code);
#endif

  if (code == 200) {
    String p = http.getString();
    dbg("RAW_TIME_JSON", p);

    earthTime = extractValue(p, "earth_time");
    marsTime  = extractValue(p, "mars_time");

    dbg("earthTime(raw)", earthTime);
    dbg("marsTime(raw)", marsTime);
    dbg("earthTime(formatted)", formatEarthTime(earthTime));
  }
  http.end();
}

void fetchDHT() {
  HTTPClient http;
  http.begin(DHT_API);
  int code = http.GET();

#if DEBUG_API
  Serial.println("\n=== DHT API ===");
  Serial.print("HTTP Code: ");
  Serial.println(code);
#endif

  if (code == 200) {
    String p = http.getString();
    dbg("RAW_DHT_JSON", p);

    temperature = extractFloat(p, "temperature");
    humidity    = extractFloat(p, "humidity");

    dbgFloat("temperature", temperature);
    dbgFloat("humidity", humidity);
  }
  http.end();
}

void fetchCompanion() {
  HTTPClient http;
  http.begin(COMPANION_API);
  http.addHeader("Content-Type", "application/json");

  String payload = "{";
  payload += "\"temperature\":" + String(temperature, 1) + ",";
  payload += "\"humidity\":" + String(humidity, 1) + ",";
  payload += "\"earth_time\":\"" + formatEarthTime(earthTime) + "\",";
  payload += "\"mars_time\":\"" + marsTime + "\",";
  payload += "\"user_input\":\"" + userInput + "\"";
  payload += "}";

#if DEBUG_API
  Serial.println("\n=== COMPANION API ===");
  dbg("REQUEST_PAYLOAD", payload);
#endif

  int code = http.POST(payload);

#if DEBUG_API
  Serial.print("HTTP Code: ");
  Serial.println(code);
#endif

  if (code == 200) {
    String res = http.getString();
    dbg("RAW_COMPANION_RESPONSE", res);

    // Extract inner escaped JSON
    String inner = extractObject(res, "response");
    dbg("INNER_JSON_ESCAPED", inner);

    // 🔥 CRITICAL FIX: unescape \" → "
    inner.replace("\\\"", "\"");
    inner.replace("\\n", "");
    dbg("INNER_JSON_CLEAN", inner);

    String action = extractRaw(inner, "action");
    String prompt = extractRaw(inner, "prompt_text");

    dbg("action", action);
    dbg("prompt_text", prompt);

    if (action == "PROMPT") {
      activePrompt = prompt;
      promptActive = true;
      drawPrompt(activePrompt);
    } else {
      promptActive = false;
      currentScreen = 0;
      drawOLED();
    }
  }
  else {
    promptActive=false;
  }
  userInput = "none";
  http.end();
}

// ---------------- TOUCH HANDLER ----------------
void handleTouch() {
  if (touch.isLeftTouched()) {
    dbg("TOUCH", "LEFT");
    userInput = "left";
    currentScreen = currentScreen - 1;
    if (currentScreen > 3) {
      currentScreen = 3;
    }
    drawOLED();
    delay(300);
  }

  if (touch.isMiddleTouched()) {
    dbg("TOUCH", "MIDDLE");
    userInput = "middle";
    currentScreen = 0;
    drawOLED();
    delay(300);
  }

  if (touch.isRightTouched()) {
    dbg("TOUCH", "RIGHT");
    userInput = "right";
    currentScreen = (currentScreen + 1) % 4;
    drawOLED();
    delay(300);
  }
}

String handleReply() {
  if (touch.isLeftTouched()) {
    dbg("TOUCH", "YES");
    userInput = "YES";
    currentScreen = 4;
    drawOLED();
    delay(300);
    return "YES";
  }

  if (touch.isMiddleTouched()) {
    dbg("TOUCH", "SOS");
    userInput = "SOS";
    currentScreen = 6;
    drawOLED();
    delay(300);
    return "SOS";
  }

  if (touch.isRightTouched()) {
    dbg("TOUCH", "NO");
    userInput = "NO";
    currentScreen = 5;
    drawOLED();
    delay(300);
    return "NO";
  }

  return "none";
}

// ---------------- SETUP ----------------
void setup() {
  Serial.begin(115200);
  Wire.begin();

  // OLED
  if (!display.begin(SSD1306_SWITCHCAPVCC, OLED_ADDR)) {
    while (1);
  }

// Setup sensor
 if (touch.begin() == false)
 {
   Serial.println("Not connected. Please check connections and read the hookup guide.");
   while (1)
     ;
 }
 else
 {
   Serial.println("Connected!");
 }

  // WiFi
  display.clearDisplay();
  display.setCursor(0, 0);
  display.println("Connecting WiFi...");
  display.display();

  WiFi.begin(ssid, password);
  while (WiFi.status() != WL_CONNECTED) delay(300);

  display.clearDisplay();
  display.setCursor(0, 10);
  display.println("WiFi Connected");
  display.display();
  delay(1000);

  fetchTime();
  fetchDHT();
  drawOLED();
}

void loop() {
  unsigned long now = millis();

  // Companion engine poll
  if ((now - lastCompanionPoll > COMPANION_INTERVAL) || (userInput == "YES") || (userInput == "NO")) {
    lastCompanionPoll = now;
    fetchCompanion();
    if (promptActive) {
      replyReceived = handleReply();
      while (replyReceived == "none") {
        Serial.println("waiting for reply");
        replyReceived = handleReply();
        delay(300);
      }
      if (replyReceived == "YES") {
        Serial.println("YES reply received, proceeding with YES");
      }
      else if (replyReceived == "NO") {
        Serial.println("NO reply received, proceeding with NO");
      }
      else if (replyReceived == "SOS") {
        // startBuzzer();
        // while ! (MiddleButtonPressed()) {
        //   delay(200);
        // }
        // stopBuzzer();
        userInput = "none";
        promptActive=false;
      }
    }


  }

  // Normal operation ONLY if no prompt
  if (!promptActive) {
    handleTouch();

    if ((now - lastScreenChange > SCREEN_CHANGE_INTERVAL)) {
      lastScreenChange = now;
      currentScreen+=1;
      if (currentScreen > 3) {
        currentScreen = 0;
      }
      drawOLED();
    }
  }


  delay(400);
}